/* =====================================================
   ISF 10+2 Simulator - Main Application Logic
   Aduanalibre.com Educational Tool
   ===================================================== */

// Global ISF Data Object
let isfData = {};

// Initialize app on page load
document.addEventListener('DOMContentLoaded', function() {
    console.log('ISF 10+2 Simulator initialized');
    setupFormValidation();
    trackAnalytics('page_view', 'ISF Simulator Loaded');
});

// Form Validation Setup
function setupFormValidation() {
    const form = document.getElementById('isfForm');
    
    // Real-time validation
    const inputs = form.querySelectorAll('input, select');
    inputs.forEach(input => {
        input.addEventListener('blur', function() {
            validateField(this);
        });
        
        input.addEventListener('input', function() {
            if (this.classList.contains('is-invalid')) {
                validateField(this);
            }
        });
    });
}

// Validate individual field
function validateField(field) {
    const value = field.value.trim();
    let isValid = true;
    let errorMessage = '';
    
    // Required field check
    if (field.hasAttribute('required') && !value) {
        isValid = false;
        errorMessage = 'This field is required';
    }
    
    // Pattern validation
    if (field.hasAttribute('pattern') && value) {
        const pattern = new RegExp(field.getAttribute('pattern'));
        if (!pattern.test(value)) {
            isValid = false;
            errorMessage = 'Invalid format';
        }
    }
    
    // Specific field validations
    if (field.id === 'importer_number' || field.id === 'consignee_number') {
        if (value && !/^\d{2}-\d{7}$/.test(value)) {
            isValid = false;
            errorMessage = 'Format must be XX-XXXXXXX (e.g., 12-3456789)';
        }
    }
    
    if (field.id === 'hts_number') {
        if (value && !/^\d{4}\.\d{2}/.test(value)) {
            isValid = false;
            errorMessage = 'HTS must be at least 6 digits (XXXX.XX)';
        }
    }
    
    // Update UI
    if (isValid) {
        field.classList.remove('is-invalid');
        field.classList.add('is-valid');
        removeError(field);
    } else {
        field.classList.remove('is-valid');
        field.classList.add('is-invalid');
        showError(field, errorMessage);
    }
    
    return isValid;
}

// Show error message
function showError(field, message) {
    removeError(field);
    const errorDiv = document.createElement('div');
    errorDiv.className = 'invalid-feedback';
    errorDiv.textContent = typeof t === 'function' ? t(message) : message;
    field.parentNode.appendChild(errorDiv);
}

// Remove error message
function removeError(field) {
    const existingError = field.parentNode.querySelector('.invalid-feedback');
    if (existingError) {
        existingError.remove();
    }
}

// Validate entire form
function validateForm() {
    const form = document.getElementById('isfForm');
    const inputs = form.querySelectorAll('input[required], select[required]');
    let allValid = true;
    let firstInvalidField = null;
    
    inputs.forEach(input => {
        const isValid = validateField(input);
        if (!isValid) {
            allValid = false;
            if (!firstInvalidField) {
                firstInvalidField = input;
            }
        }
    });
    
    // Scroll to first invalid field
    if (!allValid && firstInvalidField) {
        firstInvalidField.scrollIntoView({ 
            behavior: 'smooth', 
            block: 'center' 
        });
        firstInvalidField.focus();
    }
    
    return allValid;
}

// Collect form data
function collectFormData() {
    isfData = {
        metadata: {
            generated_at: new Date().toISOString(),
            generated_by: 'Aduanalibre.com ISF Simulator',
            version: '1.0',
            disclaimer: 'EDUCATIONAL TOOL ONLY - NOT TRANSMITTED TO CBP'
        },
        vessel_information: {
            vessel_name: document.getElementById('vessel_name').value,
            bill_of_lading: document.getElementById('bill_of_lading').value,
            loading_date: document.getElementById('loading_date').value,
            estimated_arrival_date: document.getElementById('arrival_date').value,
            port_of_unlading: document.getElementById('port_unlading').value
        },
        isf_elements: {
            element_1_seller: {
                name: document.getElementById('seller_name').value,
                address: document.getElementById('seller_address').value,
                city: document.getElementById('seller_city').value,
                state: document.getElementById('seller_state').value,
                country: document.getElementById('seller_country').value
            },
            element_2_buyer: {
                name: document.getElementById('buyer_name').value,
                address: document.getElementById('buyer_address').value,
                city: document.getElementById('buyer_city').value,
                state: document.getElementById('buyer_state').value,
                zip: document.getElementById('buyer_zip').value,
                country: 'US'
            },
            element_3_importer: {
                number: document.getElementById('importer_number').value,
                name: document.getElementById('importer_name').value
            },
            element_4_consignee: {
                number: document.getElementById('consignee_number').value,
                name: document.getElementById('consignee_name').value
            },
            element_5_manufacturer: {
                name: document.getElementById('manufacturer_name').value,
                address: document.getElementById('manufacturer_address').value,
                country: document.getElementById('manufacturer_country').value,
                flexible: true,
                note: 'Can be updated before vessel arrival'
            },
            element_6_ship_to: {
                name: document.getElementById('ship_to_name').value,
                address: document.getElementById('ship_to_address').value,
                city: document.getElementById('ship_to_city').value,
                state: document.getElementById('ship_to_state').value,
                zip: document.getElementById('ship_to_zip').value,
                country: 'US',
                flexible: true,
                note: 'First delivery location in USA'
            },
            element_7_country_of_origin: {
                country: document.getElementById('country_origin').value,
                flexible: true,
                note: 'Where goods were manufactured'
            },
            element_8_hts: {
                code: document.getElementById('hts_number').value,
                description: document.getElementById('commodity_description').value,
                flexible: true,
                note: 'Minimum 6 digits required'
            },
            element_9_stuffing_location: {
                location: document.getElementById('stuffing_location').value,
                timing: 'Required 24 hours before vessel ARRIVAL',
                note: 'Where container was loaded'
            },
            element_10_consolidator: {
                name: document.getElementById('consolidator_name').value,
                address: document.getElementById('consolidator_address').value,
                timing: 'Required 24 hours before vessel ARRIVAL',
                note: 'Who stuffed/loaded the container'
            }
        },
        compliance: {
            filing_deadline: '24 hours before vessel loading at foreign port',
            update_deadline: '24 hours before vessel arrival at US port',
            penalty_for_violation: '$5,000 USD per violation',
            responsible_party: 'ISF Importer or their authorized agent'
        },
        next_steps: {
            step_1: 'Validate this data with your customs broker',
            step_2: 'File through ACE Portal (if <12 filings/year) or certified software',
            step_3: 'Obtain ISF Bond ($10,000 minimum)',
            step_4: 'Receive confirmation number from CBP after successful filing',
            step_5: 'Update flexible elements before vessel arrival if needed'
        }
    };
    
    return isfData;
}

// Main validation and export function
function validateAndExport() {
    trackAnalytics('button_click', 'Validate and Export Clicked');
    
    if (!validateForm()) {
        const msg = typeof t === 'function' ? t('validation_error') : '⚠️ Please complete all required fields correctly before exporting.';
        alert(msg);
        return;
    }
    
    // Collect data
    collectFormData();
    
    // Show success message
    showSuccessAnimation();
    
    // Show export section
    document.getElementById('exportSection').style.display = 'block';
    document.getElementById('exportSection').scrollIntoView({ 
        behavior: 'smooth' 
    });
    
    trackAnalytics('validation_success', 'Form Validated Successfully');
}

// Success animation
function showSuccessAnimation() {
    const btn = event.target;
    const originalHTML = btn.innerHTML;
    
    const successMsg = typeof t === 'function' ? t('validated_success') : 'Validated Successfully!';
    btn.innerHTML = `<i class="fas fa-check-circle"></i> ${successMsg}`;
    btn.classList.add('btn-success');
    btn.disabled = true;
    
    setTimeout(() => {
        btn.innerHTML = originalHTML;
        btn.disabled = false;
    }, 3000);
}

// Show info modal
function showInfoModal() {
    const modal = new bootstrap.Modal(document.getElementById('helpModal'));
    document.getElementById('helpModalTitle').textContent = 'About ISF 10+2';
    document.getElementById('helpModalContent').innerHTML = `
        <h4>What is ISF 10+2?</h4>
        <p><strong>ISF (Importer Security Filing)</strong>, also known as "10+2", is a U.S. Customs and Border Protection (CBP) security regulation that requires importers to submit specific cargo information electronically.</p>
        
        <h5>The "10+2" Breakdown:</h5>
        <ul>
            <li><strong>10 elements</strong> - Provided by the importer</li>
            <li><strong>2 elements</strong> - Provided by the ocean carrier</li>
        </ul>
        
        <h5>Key Requirements:</h5>
        <ul>
            <li>✅ Applies ONLY to ocean cargo arriving in the USA</li>
            <li>✅ Must be filed 24 hours BEFORE cargo is loaded onto vessel</li>
            <li>✅ Penalty: $5,000 per violation</li>
            <li>✅ Can be filed by importer or licensed customs broker</li>
        </ul>
        
        <h5>⚠️ Important:</h5>
        <p class="text-danger"><strong>This simulator is for educational purposes only.</strong> It does NOT file actual ISF to CBP. You must use a licensed customs broker or certified platform for official filing.</p>
    `;
    modal.show();
}

// Show full disclaimer
function showFullDisclaimer() {
    const modal = new bootstrap.Modal(document.getElementById('helpModal'));
    document.getElementById('helpModalTitle').textContent = '⚠️ Legal Disclaimer';
    document.getElementById('helpModalContent').innerHTML = `
        <div class="alert alert-warning">
            <h4>EDUCATIONAL TOOL ONLY - READ CAREFULLY</h4>
        </div>
        
        <h5>What This Tool IS:</h5>
        <ul>
            <li>✅ Educational simulator for learning ISF requirements</li>
            <li>✅ Practice tool for understanding CBP data elements</li>
            <li>✅ Data preparation assistant</li>
            <li>✅ Training resource for import/export professionals</li>
        </ul>
        
        <h5>What This Tool IS NOT:</h5>
        <ul>
            <li>❌ NOT a customs broker or licensed filing agent</li>
            <li>❌ NOT connected to CBP systems (ACE/ABI)</li>
            <li>❌ NOT transmitting data to U.S. Customs</li>
            <li>❌ NOT providing legal or compliance advice</li>
        </ul>
        
        <h5>Your Responsibilities:</h5>
        <ol>
            <li><strong>Validate:</strong> All data must be verified by you or your customs broker</li>
            <li><strong>File Officially:</strong> Use ACE Portal, certified software, or hire a customs broker</li>
            <li><strong>Obtain ISF Bond:</strong> Required $10,000 minimum bond</li>
            <li><strong>Meet Deadlines:</strong> File 24 hours before loading, update 24 hours before arrival</li>
            <li><strong>Accept Liability:</strong> You are responsible for accuracy and timeliness</li>
        </ol>
        
        <h5>Penalties for Non-Compliance:</h5>
        <ul>
            <li>💰 $5,000 fine per violation</li>
            <li>⏸️ Cargo hold/delay</li>
            <li>🚫 Denied permit to unlade</li>
            <li>📦 Possible seizure of goods</li>
        </ul>
        
        <p class="mt-4"><strong>By using this tool, you acknowledge that:</strong></p>
        <ul>
            <li>This is NOT official ISF filing</li>
            <li>Aduanalibre.com is NOT a customs broker</li>
            <li>You must file through proper channels</li>
            <li>You are solely responsible for compliance</li>
        </ul>
    `;
    modal.show();
}

// Coming Soon Modal
function showComingSoon() {
    trackAnalytics('feature_interest', 'EDI Coming Soon Modal Opened');
    document.getElementById('comingSoonModal').classList.add('show');
}

function closeComingSoonModal() {
    document.getElementById('comingSoonModal').classList.remove('show');
}

// Waitlist submission
function joinWaitlist(event) {
    event.preventDefault();
    
    const email = document.getElementById('waitlist_email').value;
    const urgency = document.getElementById('urgency').value;
    
    // Simulate API call (you'll need to implement actual backend)
    console.log('Waitlist submission:', { email, urgency });
    
    // Track analytics
    trackAnalytics('waitlist_join', 'EDI Waitlist Signup', {
        urgency: urgency,
        email: email
    });
    
    // Show success message
    document.getElementById('waitlistForm').style.display = 'none';
    document.getElementById('waitlistSuccess').style.display = 'block';
    
    // You would implement actual API call here:
    /*
    fetch('/api/waitlist/edi', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email, urgency, timestamp: new Date().toISOString() })
    })
    .then(response => response.json())
    .then(data => {
        console.log('Waitlist saved:', data);
    });
    */
    
    // Close modal after 3 seconds
    setTimeout(() => {
        closeComingSoonModal();
        // Reset form
        document.getElementById('waitlistForm').style.display = 'block';
        document.getElementById('waitlistSuccess').style.display = 'none';
        document.getElementById('waitlistForm').reset();
    }, 3000);
}

// Analytics tracking (Google Analytics 4 compatible)
function trackAnalytics(eventName, eventLabel, customData = {}) {
    if (typeof gtag !== 'undefined') {
        gtag('event', eventName, {
            'event_category': 'isf_simulator',
            'event_label': eventLabel,
            ...customData
        });
    }
    
    // Console log for development
    console.log(`Analytics: ${eventName} - ${eventLabel}`, customData);
}

// Close modals when clicking outside
window.onclick = function(event) {
    const comingSoonModal = document.getElementById('comingSoonModal');
    if (event.target === comingSoonModal) {
        closeComingSoonModal();
    }
}

// Reset form handler
document.getElementById('isfForm').addEventListener('reset', function() {
    // Clear validation classes
    const inputs = this.querySelectorAll('input, select');
    inputs.forEach(input => {
        input.classList.remove('is-valid', 'is-invalid');
        removeError(input);
    });
    
    // Hide export section
    document.getElementById('exportSection').style.display = 'none';
    
    // Scroll to top
    window.scrollTo({ top: 0, behavior: 'smooth' });
    
    trackAnalytics('form_reset', 'Form Cleared');
});
