/* =====================================================
   ISF 10+2 Simulator - Language System
   English / Spanish Translation System
   ===================================================== */

// Default language
let currentLanguage = 'en';

// Load saved language preference on page load
document.addEventListener('DOMContentLoaded', function() {
    const savedLang = localStorage.getItem('isf_simulator_language');
    if (savedLang && (savedLang === 'en' || savedLang === 'es')) {
        changeLanguage(savedLang, false);
    }
    console.log('Language system initialized:', currentLanguage);
});

// Change language function
function changeLanguage(lang, savePreference = true) {
    if (lang !== 'en' && lang !== 'es') {
        console.error('Invalid language:', lang);
        return;
    }
    
    currentLanguage = lang;
    
    // Update HTML lang attribute
    document.getElementById('html-root').setAttribute('lang', lang);
    
    // Update all elements with data-en and data-es attributes
    const elements = document.querySelectorAll('[data-en][data-es]');
    elements.forEach(element => {
        const translation = element.getAttribute(`data-${lang}`);
        if (translation) {
            // Check if it's an input placeholder
            if (element.tagName === 'INPUT' || element.tagName === 'TEXTAREA') {
                element.placeholder = translation;
            } else {
                element.textContent = translation;
            }
        }
    });
    
    // Update language toggle buttons
    document.querySelectorAll('.lang-btn').forEach(btn => {
        if (btn.getAttribute('data-lang') === lang) {
            btn.classList.add('active');
        } else {
            btn.classList.remove('active');
        }
    });
    
    // Save preference
    if (savePreference) {
        localStorage.setItem('isf_simulator_language', lang);
        if (typeof trackAnalytics === 'function') {
            trackAnalytics('language_change', `Language changed to ${lang.toUpperCase()}`);
        }
    }
    
    console.log('Language changed to:', lang);
}

// Get current language
function getCurrentLanguage() {
    return currentLanguage;
}

// Translation helper for JavaScript strings
const translations = {
    en: {
        // Success messages
        'validation_success': 'Form validated successfully!',
        'download_success': 'Downloaded!',
        'waitlist_success': 'Great! We\'ll notify you when EDI export is available.',
        
        // Error messages
        'validation_error': '⚠️ Please complete all required fields correctly before exporting.',
        'field_required': 'This field is required',
        'invalid_format': 'Invalid format',
        'ein_format_error': 'Format must be XX-XXXXXXX (e.g., 12-3456789)',
        'hts_format_error': 'HTS must be at least 6 digits (XXXX.XX)',
        
        // Buttons
        'validate_export': 'Validate & Export ISF Data',
        'validated_success': 'Validated Successfully!',
        'clear_form': 'Clear Form',
        'download_json': 'Download JSON',
        'download_excel': 'Download Excel',
        'download_csv': 'Download CSV',
        'notify_me': 'Notify Me When Ready',
        'close': 'Close'
    },
    es: {
        // Success messages
        'validation_success': '¡Formulario validado con éxito!',
        'download_success': '¡Descargado!',
        'waitlist_success': '¡Genial! Te notificaremos cuando la exportación EDI esté disponible.',
        
        // Error messages
        'validation_error': '⚠️ Por favor complete todos los campos requeridos correctamente antes de exportar.',
        'field_required': 'Este campo es requerido',
        'invalid_format': 'Formato inválido',
        'ein_format_error': 'El formato debe ser XX-XXXXXXX (ej: 12-3456789)',
        'hts_format_error': 'El código HTS debe tener al menos 6 dígitos (XXXX.XX)',
        
        // Buttons
        'validate_export': 'Validar y Exportar Datos ISF',
        'validated_success': '¡Validado con Éxito!',
        'clear_form': 'Limpiar Formulario',
        'download_json': 'Descargar JSON',
        'download_excel': 'Descargar Excel',
        'download_csv': 'Descargar CSV',
        'notify_me': 'Notificarme Cuando Esté Listo',
        'close': 'Cerrar'
    }
};

// Get translation for a key
function t(key) {
    return translations[currentLanguage][key] || translations.en[key] || key;
}
